<?php

if (!defined('WHMCS')) {
    die('This file cannot be accessed directly');
}

use WHMCS\Database\Capsule;

/**
 * Module metadata.
 *
 * @return array
 */
function multivisionappsubscription_MetaData()
{
    return [
        'DisplayName' => 'Multivision App Subscription',
        'APIVersion' => '1.1',
        'RequiresServer' => false,
    ];
}

/**
 * Module configuration options.
 *
 * @return array
 */
function multivisionappsubscription_ConfigOptions()
{
    $pid = $_REQUEST['id'] ?? null;
    // Ensure MacAddress custom field exists for products using this module.
    multivisionappsubscription_ensureMacAddressCustomFieldForProduct($pid);

    return [];
}

/**
 * Provision a service.
 *
 * @param array $params
 *
 * @return string
 */
function multivisionappsubscription_CreateAccount(array $params)
{
    return multivisionappsubscription_syncSubscription($params, "active");
}

/**
 * Suspend a service.
 *
 * @param array $params
 *
 * @return string
 */
function multivisionappsubscription_SuspendAccount(array $params)
{
    return multivisionappsubscription_syncSubscription($params, 'suspended');
}

/**
 * Unsuspend a service.
 *
 * @param array $params
 *
 * @return string
 */
function multivisionappsubscription_UnsuspendAccount(array $params)
{
    return multivisionappsubscription_syncSubscription($params, 'active');
}

/**
 * Terminate a service.
 *
 * @param array $params
 *
 * @return string
 */
function multivisionappsubscription_TerminateAccount(array $params)
{
    return multivisionappsubscription_syncSubscription($params, 'terminated');
}

/*
 * Upgrade or downgrade a service.
 *
 * @param array $params
 *
 * @return string
 */
function multivisionappsubscription_ChangePackage(array $params)
{
    return multivisionappsubscription_syncSubscription($params);
}


function multivisionappsubscription_Renew(array $params)
{
    try {
        $macAddress = multivisionappsubscription_getMacAddress($params);
        if ($macAddress === '') {
            logModuleCall(
                'multivisionappsubscription',
                'renewSubscription',
                multivisionappsubscription_buildLogPayload($params, 'active'),
                ['result' => 'success', 'message' => 'Skipped renew because MacAddress custom field is empty']
            );

            return 'success';
        }

        $billingCycle = isset($params['model']->billingcycle) ? (string) $params['model']->billingcycle : '';
        $intervalSpec = multivisionappsubscription_getRenewIntervalSpec($billingCycle);
        if ($intervalSpec === '') {
            $response = [
                'result' => 'error',
                'message' => 'Unsupported billing cycle for renewal: ' . $billingCycle,
            ];

            logModuleCall(
                'multivisionappsubscription',
                'renewSubscription',
                multivisionappsubscription_buildLogPayload($params, 'active'),
                $response
            );

            return $response['message'];
        }

        $device = Capsule::table('tbl_devices')
            ->where('mac_id', $macAddress)
            ->first();

        if (!$device) {
            $response = [
                'result' => 'success',
                'message' => 'No matching device found, renew skipped',
            ];

            logModuleCall(
                'multivisionappsubscription',
                'renewSubscription',
                [
                    'macAddress' => $macAddress,
                    'billingcycle' => $billingCycle,
                    'userid' => isset($params['userid']) ? $params['userid'] : null,
                    'pid' => isset($params['pid']) ? $params['pid'] : null,
                ],
                $response
            );

            return 'success';
        }

        $subscription = Capsule::table('tbl_subscriptions')
            ->where('device_id', $device->id)
            ->first();

        $currentExpiryDate = '';
        if ($subscription && !empty($subscription->expiry_date) && $subscription->expiry_date !== '0000-00-00') {
            $currentExpiryDate = (string) $subscription->expiry_date;
        } else {
            $service = isset($params['model']) ? $params['model'] : null;
            if ($service && isset($service->nextduedate)) {
                $currentExpiryDate = (string) $service->nextduedate;
            } elseif (isset($params['nextduedate'])) {
                $currentExpiryDate = (string) $params['nextduedate'];
            }
        }

        $baseDate = multivisionappsubscription_getRenewBaseDate($currentExpiryDate);
        $newExpiryDate = $baseDate->add(new DateInterval($intervalSpec))->format('Y-m-d');

        Capsule::table('tbl_devices')
            ->where('id', $device->id)
            ->update([
                'user_id' => isset($params['userid']) ? (int) $params['userid'] : 0,
                'updated_at' => date('Y-m-d H:i:s'),
            ]);

        Capsule::table('tbl_subscriptions')
            ->where('device_id', $device->id)
            ->update([
                'product_id' => isset($params['pid']) ? (int) $params['pid'] : 0,
                'package_type' => $billingCycle,
                'status' => 'active',
                'expiry_date' => $newExpiryDate,
            ]);

        logModuleCall(
            'multivisionappsubscription',
            'renewSubscription',
            [
                'device_id' => $device->id,
                'macAddress' => $macAddress,
                'billingcycle' => $billingCycle,
                'interval' => $intervalSpec,
                'previous_expiry_date' => $currentExpiryDate,
            ],
            [
                'result' => 'success',
                'new_expiry_date' => $newExpiryDate,
            ]
        );

        return 'success';
    } catch (\Throwable $e) {
        logModuleCall(
            'multivisionappsubscription',
            'renewSubscriptionException',
            multivisionappsubscription_buildLogPayload($params, 'active'),
            ['exception' => $e->getMessage()]
        );

        return 'Unhandled exception: ' . $e->getMessage();
    }
}

/**
 * Apply the same table updates that were previously executed by hooks.
 *
 * @param array       $params
 * @param string|null $forcedStatus
 *
 * @return string success|error message
 */
function multivisionappsubscription_syncSubscription(array $params, $forcedStatus = null)
{
    try {
        $macAddress = multivisionappsubscription_getMacAddress($params);

        if ($macAddress === '') {
            $response = [
                'result' => 'error',
                'message' => 'Skipped sync because MacAddress custom field is empty',
            ];

            logModuleCall(
                'multivisionappsubscription',
                'syncSubscription',
                multivisionappsubscription_buildLogPayload($params, $forcedStatus),
                $response
            );

            return 'MacAddress is empty, sync skipped';
        }

        $service = isset($params['model']) ? $params['model'] : null;
        $nextDueDate = '';

        if ($service && isset($service->nextduedate)) {
            $nextDueDate = (string) $service->nextduedate;
        } elseif (isset($params['nextduedate'])) {
            $nextDueDate = (string) $params['nextduedate'];
        }
        $billingCycle =  $params['model']->billingcycle;
        $productId = isset($params['pid']) ? (int) $params['pid'] : 0;
        $userId = isset($params['userid']) ? (int) $params['userid'] : 0;

        $rawStatus = '';
        if ($forcedStatus !== null) {
            $rawStatus = $forcedStatus;
        } elseif (isset($params['status']) && $params['status'] !== '') {
            $rawStatus = (string) $params['status'];
        } else {
            $rawStatus = 'active';
        }

        $status = strtolower($rawStatus);

        $requestPayload = [
            'userid' => $userId,
            'productId' => $productId,
            'nextDueDate' => ($billingCycle == 'Free Account' || $billingCycle == 'One Time') ? null : $nextDueDate,
            'billingcycle' => $billingCycle,
            'MacAddress' => $macAddress,
            'status' => $status,
        ];
        // print_r($requestPayload); exit;
        $response = multivisionappsubscription_updateCustomTables($requestPayload);

        logModuleCall(
            'multivisionappsubscription',
            'syncSubscription',
            $requestPayload,
            $response
        );

        if ($response['result'] === 'success') {
            return 'success';
        }

        return $response['message'];
    } catch (\Throwable $e) {
        logModuleCall(
            'multivisionappsubscription',
            'syncSubscriptionException',
            multivisionappsubscription_buildLogPayload($params, $forcedStatus),
            ['exception' => $e->getMessage()]
        );

        return 'Unhandled exception: ' . $e->getMessage();
    }
}

/**
 * Update tbl_devices and tbl_subscriptions based on the Mac Address.
 *
 * @param array $data
 *
 * @return array
 */
function multivisionappsubscription_updateCustomTables(array $data)
{
    try {
        $response = [
            'result' => 'error',
            'message' => 'Unable to update data',
        ];

        $device = Capsule::table('tbl_devices')
            ->where('mac_id', $data['MacAddress'])
            ->first();

        if (!$device) {
            return [
                'result' => 'success',
                'message' => 'No matching device found, sync skipped',
                'skipped' => true,
            ];
        }

        Capsule::table('tbl_devices')
            ->where('id', $device->id)
            ->update([
                'user_id' => $data['userid'],
                'updated_at' => date('Y-m-d H:i:s'),
            ]);

        Capsule::table('tbl_subscriptions')
            ->where('device_id', $device->id)
            ->update([
                'product_id' => $data['productId'],
                'package_type' => $data['billingcycle'],
                'status' => $data['status'],
                'expiry_date' => $data['nextDueDate'],
            ]);

        return [
            'result' => 'success',
            'message' => 'Subscription updated successfully',
         ];
    } catch (\Throwable $th) {
        return [
            'result' => 'error',
            'message' => 'Exception occurred: ' . $th->getMessage(),
        ];
    }
}

/**
 * Locate MacAddress from WHMCS custom fields, supporting common key variants.
 *
 * @param array $params
 *
 * @return string
 */
function multivisionappsubscription_getMacAddress(array $params)
{
    $customFields = isset($params['customfields']) && is_array($params['customfields'])
        ? $params['customfields']
        : [];

    if (isset($customFields['MacAddress']) && trim((string) $customFields['MacAddress']) !== '') {
        return trim((string) $customFields['MacAddress']);
    }

    $normalizedNeedle = 'macaddress';

    foreach ($customFields as $name => $value) {
        $normalizedName = strtolower(preg_replace('/[^a-z0-9]/i', '', (string) $name));

        if ($normalizedName === $normalizedNeedle && trim((string) $value) !== '') {
            return trim((string) $value);
        }
    }

    return '';
}


/**
 * Ensure a MacAddress custom field exists for a specific product.
 *
 * @param int $productId
 *
 * @return void
 */
function multivisionappsubscription_ensureMacAddressCustomFieldForProduct($productId)
{
    if ((int) $productId <= 0) {
        return;
    }

    $existingFields = Capsule::table('tblcustomfields')
        ->where('type', 'product')
        ->where('relid', (int) $productId)
        ->get(['id', 'fieldname']);

    foreach ($existingFields as $field) {
        $normalizedFieldName = strtolower(preg_replace('/[^a-z0-9]/i', '', (string) $field->fieldname));
        if ($normalizedFieldName === 'macaddress') {
            return;
        }
    }

    $maxSortOrder = Capsule::table('tblcustomfields')
        ->where('type', 'product')
        ->where('relid', (int) $productId)
        ->max('sortorder');

    $sortOrder = is_numeric($maxSortOrder) ? ((int) $maxSortOrder + 1) : 0;

    $insertData = multivisionappsubscription_buildCustomFieldInsertData((int) $productId, $sortOrder);
    Capsule::table('tblcustomfields')->insert($insertData);
}

/**
 * Build insert payload for tblcustomfields, adapting to available columns.
 *
 * @param int $productId
 * @param int $sortOrder
 *
 * @return array
 */
function multivisionappsubscription_buildCustomFieldInsertData($productId, $sortOrder)
{
    $insertData = [
        'type' => 'product',
        'relid' => (int) $productId,
        'fieldname' => 'MacAddress',
        'fieldtype' => 'text',
        'description' => '',
        'fieldoptions' => '',
        'regexpr' => '',
        'adminonly' => '',
        'required' => 'on',
        'showorder' => 'on',
        'showinvoice' => '',
        'sortorder' => (int) $sortOrder,
    ];

    try {
        $schema = Capsule::schema();
        foreach (array_keys($insertData) as $column) {
            if (in_array($column, ['type', 'relid', 'fieldname', 'fieldtype'], true)) {
                continue;
            }

            if (!$schema->hasColumn('tblcustomfields', $column)) {
                unset($insertData[$column]);
            }
        }
    } catch (\Throwable $e) {
        // Fall back to the full insert payload.
    }

    return $insertData;
}

/**
 * Resolve billing cycle to DateInterval ISO spec for renew operation.
 *
 * @param string $billingCycle
 *
 * @return string
 */
function multivisionappsubscription_getRenewIntervalSpec($billingCycle)
{
    $normalized = strtolower(trim((string) $billingCycle));

    $intervalMap = [
        'monthly' => 'P1M',
        'quarterly' => 'P3M',
        'semi-annually' => 'P6M',
        'semiannually' => 'P6M',
        'annually' => 'P1Y',
        'biennially' => 'P2Y',
        'triennially' => 'P3Y',
    ];

    return isset($intervalMap[$normalized]) ? $intervalMap[$normalized] : '';
}

/**
 * Pick renewal start date as max(today, current expiry date).
 *
 * @param string $currentExpiryDate
 *
 * @return DateTimeImmutable
 */
function multivisionappsubscription_getRenewBaseDate($currentExpiryDate)
{
    $today = new DateTimeImmutable('today');
    $value = trim((string) $currentExpiryDate);

    if ($value === '' || $value === '0000-00-00') {
        return $today;
    }

    try {
        $date = new DateTimeImmutable($value);
    } catch (\Throwable $e) {
        return $today;
    }

    $date = $date->setTime(0, 0, 0);

    return $date < $today ? $today : $date;
}

/**
 * Build lightweight payload for module call logs.
 *
 * @param array       $params
 * @param string|null $forcedStatus
 *
 * @return array
 */
function multivisionappsubscription_buildLogPayload(array $params, $forcedStatus)
{
    return [
        'serviceid' => isset($params['serviceid']) ? $params['serviceid'] : null,
        'userid' => isset($params['userid']) ? $params['userid'] : null,
        'pid' => isset($params['pid']) ? $params['pid'] : null,
        'status' => isset($params['status']) ? $params['status'] : null,
        'forcedStatus' => $forcedStatus,
        'customfields' => isset($params['customfields']) ? $params['customfields'] : [],
    ];
}
